package easik.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.ArrayList;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JColorChooser;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;

import easik.Easik;
import easik.EasikConstants;
import easik.sketch.constraint.Constraint;
import easik.sketch.datatype.DataType;
import easik.sketch.util.ini.IniSettings;
import easik.ui.datatype.DataTypePane;
import easik.ui.datatype.DatatypesUI;
import easik.ui.menu.XMLFileFilter;

/**
 * This dialog is used to set various program options
 * 
 * @author Kevin Green 2006
 * @since 2006-08-01 Kevin Green
 * @version 2006-08-04 Kevin Green
 */
public class ProgramSettingsUI extends JDialog{

	/**
	 * The basic settings panel (This one has components)
	 */
	private JPanel _basicPanel;
	/**
	 * The color settings panel (This one has components)
	 */
	private JPanel _colorPanel;
	/**
	 * The bottom button pane
	 */
	private JPanel _buttonPane;
	/**
	 * The "Ok" button
	 */
	private JButton _okButton;
	/**
	 * The "Cancel" button 
	 */
	private JButton _cancelButton;
	/**
	 * The basic settings tab scroll pane
	 */
	private JScrollPane _basicTab;
	/**
	 * The color settings tab scoll pane
	 */
	private JScrollPane _colorTab;
	/**
	 * The main tabbed pane
	 */
	private JTabbedPane _mainPane;
	/**
	 * The radio button group for default folder options 
	 */
	private ButtonGroup _folder;
	/**
	 * Radio button for Last Used Folder option
	 */
	private JRadioButton _folderLast;
	/**
	 * Radio button for Running Directory option
	 */
	private JRadioButton _folderRunning;
	/**
	 * Radio button for Specific Folder option
	 */
	private JRadioButton _folderSpecific;
	/**
	 * The text feild for the specific folder
	 */
	private JTextField _folderSpecificLoc;
	/**
	 * The button to set the specific folder
	 */
	private JButton _folderSpecificButton;
	/**
	 * The show all attributes and unique keys load value check box
	 */
	private JCheckBox _showAttsAndUniqueKeys;
	/**
	 * ArrayList to keep track of colorSetting panes
	 */
	private ArrayList<ColorSettingPane> _colorSettingPanes;

	
	/**
	 * Default Constructor
	 */
	public ProgramSettingsUI(){
		super(Easik.getInstance().getFrame(), "Program Settings", true);
		
		_colorSettingPanes = new ArrayList<ColorSettingPane>();
		
		makeDialog();
		this.setSize(new Dimension(500, 500));
		this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
		this.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent evt) {
                cancelAction();
            }
        });
	}
	
	/**
	 * Actions to be taken when cancel button is pressed or window exited
	 * 
	 * Will revert color back and not store in other values
	 */
	public void cancelAction(){
		for(ColorSettingPane curPane : _colorSettingPanes){
			curPane.revertColor();
		}
		
		Easik.getInstance().getStateManager().selectionUpdated();
		
		boolean dirty = Easik.getInstance().getFrame().getSketch().getDirty();
		for(Constraint curConst : Easik.getInstance().getFrame().getSketch().getConstraints()){
			curConst.setVisible(!curConst.isVisible());
			curConst.setVisible(!curConst.isVisible());
		}
		Easik.getInstance().getFrame().getSketch().setDirty(dirty);
		
		this.dispose();
	}
	
	/**
	 * Actions to be taken when ok button is pressed
	 *
	 * Colors and settings will be saved to ini file
	 */
	public void saveAction(){
		IniSettings ini = Easik.getInstance().getIni();
		ini.setShowAllAttsAndUniqueKeys(_showAttsAndUniqueKeys.isSelected());
		ini.setSpecificFolder(_folderSpecificLoc.getText());
		if(_folderSpecific.isSelected()){
			ini.setSaveOpenOperation(IniSettings.FOLDER_SPECIFIC);
		}
		else if(_folderRunning.isSelected()){
			ini.setSaveOpenOperation(IniSettings.FOLDER_RUNNING);
		}
		else{
			ini.setSaveOpenOperation(IniSettings.FOLDER_LAST_USED);
		}
		
		this.dispose();
	}
	
	/**
	 * Radio Button Change Action
	 */
	public void radioChanged(){
		if(_folderSpecific.isSelected()){
			_folderSpecificLoc.setEnabled(true);
			_folderSpecificButton.setEnabled(true);
		}
		else{
			_folderSpecificLoc.setEnabled(false);
			_folderSpecificButton.setEnabled(false);
		}
	}
	
	/**
	 * Makes the dialog
	 */
	public void makeDialog(){
		_mainPane = new JTabbedPane();
		_buttonPane = new JPanel();
		this.setLayout(new BorderLayout());
		this.add(_mainPane, BorderLayout.CENTER);
		this.add(_buttonPane, BorderLayout.SOUTH);
		
		makeButtonPane();

		_basicTab = new JScrollPane();
		_mainPane.addTab("Basic Settings", null, _basicTab,
		                  "Basic program settings");
		_mainPane.setMnemonicAt(0, KeyEvent.VK_B);
		
		_colorTab = new JScrollPane();
		_mainPane.addTab("Color Settings", null, _colorTab,
		                  "User interface color settings");
		_mainPane.setMnemonicAt(1, KeyEvent.VK_C);

		makeColorPane();
		makeBasicPane();
	}
	
	/**
	 * Makes the button pane
	 */
	public void makeButtonPane(){
		_buttonPane.setLayout(new GridLayout(1,0));
		
		_okButton = new JButton("OK");
		_okButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				saveAction();
			}
		});
		
		_cancelButton = new JButton("Cancel");
		_cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				cancelAction();
			}
		});
		
		_buttonPane.add(_okButton);
		_buttonPane.add(_cancelButton);
	}
	
	/**
	 * Makes the basic settings pane
	 */
	public void makeBasicPane(){
		_basicPanel = new JPanel();
		_basicPanel.setLayout(new BoxLayout(_basicPanel, BoxLayout.PAGE_AXIS));
		_basicPanel.setBorder(new EmptyBorder(1,10,1,1));
		_basicTab.setViewportView(_basicPanel);
		
		_folderLast = new JRadioButton("Last used folder");
		_folderRunning = new JRadioButton("Running directory");
		_folderSpecific = new JRadioButton("Specific folder");
		
		if(Easik.getInstance().getIni().getSaveOpenOperation() == IniSettings.FOLDER_SPECIFIC){
			_folderSpecific.setSelected(true);
		}
		else if(Easik.getInstance().getIni().getSaveOpenOperation() == IniSettings.FOLDER_RUNNING){
			_folderRunning.setSelected(true);
		}
		else{
			_folderLast.setSelected(true);
		}
		
		_folderLast.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				radioChanged();
			}
		});
		_folderRunning.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				radioChanged();
			}
		});
		_folderSpecific.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				radioChanged();
			}
		});
		
		_folder = new ButtonGroup();
		_folder.add(_folderLast);
		_folder.add(_folderRunning);
		_folder.add(_folderSpecific);
		
		JPanel specificPane = new JPanel();
		specificPane.setBorder(new EmptyBorder(1,20,1,1));
		_folderSpecificLoc = new JTextField();
		_folderSpecificLoc.setText(Easik.getInstance().getIni().getSpecificFolder());
		_folderSpecificLoc.setColumns(25);
		_folderSpecificButton = new JButton("Browse");
		specificPane.add(_folderSpecificLoc);
		specificPane.add(_folderSpecificButton);
		specificPane.setMaximumSize(specificPane.getPreferredSize());
		specificPane.setMinimumSize(specificPane.getPreferredSize());
		
		_folderSpecificButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				File selFile;			
				JFileChooser saveDialog = new JFileChooser(Easik.getInstance().getIni().getDefaultFolder());
				saveDialog.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				saveDialog.showDialog(Easik.getInstance().getFrame(), "Select Folder");
				selFile = saveDialog.getSelectedFile();
				
				if (selFile != null) {	
					_folderSpecificLoc.setText(selFile.getAbsolutePath());
				}
			}
		});
		
		JPanel folderPane = new JPanel();
		folderPane.setLayout(new BoxLayout(folderPane, BoxLayout.PAGE_AXIS));
		folderPane.setBorder(new CompoundBorder(new EmptyBorder(1,25,1,1), new LineBorder(Color.BLACK, 1)));
		_folderLast.setAlignmentX(Component.LEFT_ALIGNMENT);
		_folderRunning.setAlignmentX(Component.LEFT_ALIGNMENT);
		_folderSpecific.setAlignmentX(Component.LEFT_ALIGNMENT);
		specificPane.setAlignmentX(Component.LEFT_ALIGNMENT);
		folderPane.add(_folderLast);
		folderPane.add(_folderRunning);
		folderPane.add(_folderSpecific);
		folderPane.add(specificPane);
		
		_showAttsAndUniqueKeys = new JCheckBox("Show all attributes & unique keys on startup");
		_showAttsAndUniqueKeys.setSelected(Easik.getInstance().getIni().isShowAllAttsAndUniqueKeys());
		
		JLabel folderLabel = new JLabel("Default Open/Save folder", JLabel.LEFT);
		JLabel spacer = new JLabel(" ");
		spacer.setAlignmentX(Component.LEFT_ALIGNMENT);
		JLabel spacer2 = new JLabel(" ");
		spacer2.setAlignmentX(Component.LEFT_ALIGNMENT);
		folderLabel.setAlignmentX(Component.LEFT_ALIGNMENT);
		folderPane.setAlignmentX(Component.LEFT_ALIGNMENT);
		_showAttsAndUniqueKeys.setAlignmentX(Component.LEFT_ALIGNMENT);
		_basicPanel.add(spacer);
		_basicPanel.add(folderLabel);
		_basicPanel.add(folderPane);
		_basicPanel.add(spacer2);
		_basicPanel.add(_showAttsAndUniqueKeys);
		
		radioChanged();
	}
	
	/**
	 * Makes the color tab pane
	 */
	public void makeColorPane(){
		_colorPanel = new JPanel();
		_colorPanel.setLayout(new BoxLayout(_colorPanel, BoxLayout.PAGE_AXIS));
		_colorTab.setViewportView(_colorPanel);
		
		ColorSettingPane tempPane;
		
		/**
		 * The color of the border of entities
		 */
		tempPane = new ColorSettingPane("Entity Border"){
			public Color getColor(){
				return Easik.getInstance().getIni().getENTITY_BORDER_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setENTITY_BORDER_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of the background of entities
		 */
		tempPane = new ColorSettingPane("Entity Background"){
			public Color getColor(){
				return Easik.getInstance().getIni().getENTITY_BG_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setENTITY_BG_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of the foreground of entities
		 */
		tempPane = new ColorSettingPane("Entity Foreground"){
			public Color getColor(){
				return Easik.getInstance().getIni().getENTITY_FG_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setENTITY_FG_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of constraint borders
		 */
		tempPane = new ColorSettingPane("Constraint Border"){
			public Color getColor(){
				return Easik.getInstance().getIni().getCONSTRAINT_BORDER_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setCONSTRAINT_BORDER_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of the background of constraints
		 */
		tempPane = new ColorSettingPane("Constraint Background"){
			public Color getColor(){
				return Easik.getInstance().getIni().getCONSTRAINT_BG_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setCONSTRAINT_BG_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of the foreground of constraints
		 */
		tempPane = new ColorSettingPane("Constraint Foreground"){
			public Color getColor(){
				return Easik.getInstance().getIni().getCONSTRAINT_FG_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setCONSTRAINT_FG_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of a simple edge
		 */
		tempPane = new ColorSettingPane("Simple Edge"){
			public Color getColor(){
				return Easik.getInstance().getIni().getSIMPLE_EDGE_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setSIMPLE_EDGE_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of a fletched edge
		 */
		tempPane = new ColorSettingPane("Fletched Edge"){
			public Color getColor(){
				return Easik.getInstance().getIni().getFLETCHED_EDGE_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setFLETCHED_EDGE_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of a virtual edge
		 */
		tempPane = new ColorSettingPane("Virtual Edge"){
			public Color getColor(){
				return Easik.getInstance().getIni().getVIRTUAL_EDGE_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setVIRTUAL_EDGE_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The color of a virtual highlighted edge
		 */
		tempPane = new ColorSettingPane("Highlighted Virtual Edge"){
			public Color getColor(){
				return Easik.getInstance().getIni().getVIRTUAL_HIGHLIGHTED_EDGE_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setVIRTUAL_HIGHLIGHTED_EDGE_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The path selection color
		 */
		tempPane = new ColorSettingPane("Path Selection Color"){
			public Color getColor(){
				return Easik.getInstance().getIni().getPATH_SELECTION_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setPATH_SELECTION_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
		/**
		 * The standard selection color
		 */
		tempPane = new ColorSettingPane("Standard Selection Color"){
			public Color getColor(){
				return Easik.getInstance().getIni().getSTANDARD_SELECTION_COLOR();
			}
			public void setColor(Color c){
				Easik.getInstance().getIni().setSTANDARD_SELECTION_COLOR(c);
			}
		};
		_colorPanel.add(tempPane);
		_colorSettingPanes.add(tempPane);
	}
	
	/**
	 * Returns the inverse of a color
	 * 
	 * @param c The input color
	 * @return The inverse of the input color
	 */
	public Color inverseColor(Color c){
		float[] hsb = new float[3];
		Color.RGBtoHSB(255-c.getRed(), 255-c.getGreen(), 255-c.getBlue(), hsb);
		return Color.getHSBColor(hsb[0], hsb[1], hsb[2]);
	}
	
	/**
	 * Class used for each color in the colors tab
	 * 
	 * @author Kevin Green 2006
	 * @since 2006-08-01 Kevin Green
	 * @version 2006-08-01 Kevin Green
	 */
	private class ColorSettingPane extends JPanel{
		
		/**
		 * The name of the elements that are having their color set
		 */
		private String _colorName;
		/**
		 * The label for the GUI with the name of the elements that are having their color set
		 */
		private JLabel _colorLabel;
		/**
		 * Button used to set the color
		 */
		private JButton _colorButton;
		/**
		 * The temporary color to use for changes
		 */
		private Color _tempColor;
		/**
		 * The original color loaded when window opens
		 */
		private Color _originalColor;
		
		/**
		 * Default constructor
		 * 
		 * @param name The name of the color
		 */
		ColorSettingPane(String name){
			_colorName = name;
			_tempColor = getColor();
			_originalColor = getColor();
			_colorLabel = new JLabel("   Sample   ", JLabel.CENTER);
			_colorLabel.setOpaque(true);
			_colorLabel.setBackground(_tempColor);
			_colorLabel.setForeground(inverseColor(_tempColor));
			_colorButton = new JButton("Edit");
			_colorButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e1) {
					Color userColor = JColorChooser.showDialog(
							Easik.getInstance().getFrame(),
							"Set color",
							_tempColor);
					if(userColor != null){
						updateColor(userColor);
						_colorLabel.setBackground(getColor());
						_colorLabel.setForeground(inverseColor(_colorLabel.getBackground()));
					}
				}
			});
			this.setLayout(new GridLayout(0,3));
			this.add(new JLabel(_colorName + ": ", JLabel.RIGHT));
			this.add(_colorLabel);
			this.add(_colorButton);
			this.setSize(this.getPreferredSize());
			this.setBorder(new EmptyBorder(3,3,3,3));
		}
		
		/**
		 * Sets the color in the ini file and updates the display
		 * 
		 * @param c The new color
		 */
		public void updateColor(Color c){
			setColor(c);
			
			Easik.getInstance().getStateManager().selectionUpdated();
			if(_colorName.equals("Virtual Edge") || _colorName.equals("Highlighted Virtual Edge")){
				boolean dirty = Easik.getInstance().getFrame().getSketch().getDirty();
				for(Constraint curConst : Easik.getInstance().getFrame().getSketch().getConstraints()){
					curConst.setVisible(!curConst.isVisible());
					curConst.setVisible(!curConst.isVisible());
				}
				Easik.getInstance().getFrame().getSketch().setDirty(dirty);
			}
		}
		
		/**
		 * Sets the ini file data back to the original color
		 */
		public void revertColor(){
			setColor(_originalColor);
		}
		
		/**
		 * Returns the color associated with this pane
		 * 
		 * Note: Should be overwritten when creating
		 *
		 * @return The color associated with this pane
		 */
		public Color getColor(){return null;}
		
		/**
		 * Sets the color associated with this pane
		 * 
		 * Note: Should be overwritten when creating
		 * 
		 * @param c The color to be associated with this pane
		 */
		public void setColor(Color c){}
	}
}
